<?php
/*
Plugin Name: Meta Tags Custom
Description: Métas SEO + OG + Twitter + tags + catégories + canonical + JSON-LD (WebSite/Article) en haut du head.
Version: 2.1
Author: Mansouri Jean-Paul
*/

if (!defined('ABSPATH')) exit;

/* Donner tags + catégories aux pages */
add_action('init', function () {
    register_taxonomy_for_object_type('post_tag', 'page');
    register_taxonomy_for_object_type('category', 'page');
});

/* Impression en haut du head */
add_action('wp_head', 'mtc_print_meta_and_jsonld', 1);

function mtc_print_meta_and_jsonld() {
    global $post;

    $home_url   = esc_url(home_url('/'));
    $site_name  = get_bloginfo('name');
    $site_desc  = get_bloginfo('description');
    $locale     = 'fr_FR';

    /* DESCRIPTION GLOBALE */
    $global_desc = "Site personnel de Jean-Paul Mansouri (F4HXN) : informatique, radio, WSPR, SDR, Docker, Proxmox, monitoring, Linux, domotique.";

    /* KEYWORDS (facultatif) */
    $keywords = "radio, WSPR, SDR, DMR, Docker, Linux, Proxmox, monitoring, Grafana, Prometheus, domotique, Home Assistant, réseau";

    $is_singular = is_singular() && $post;

    /* Title / Desc / URL */
    $title = $is_singular ? get_the_title($post) : $site_name;

    $raw_excerpt = $is_singular ? wp_strip_all_tags(get_the_excerpt($post)) : '';
    $description = $raw_excerpt ? $raw_excerpt : ($site_desc ? $site_desc : $global_desc);
    if (!$description) $description = $title;

    $url = $is_singular ? get_permalink($post) : $home_url;

    /* Image OG */
    $image = '';
    if ($is_singular && has_post_thumbnail($post)) {
        $image = get_the_post_thumbnail_url($post, 'full');
    } else {
        $icon = get_site_icon_url(512);
        if ($icon) $image = $icon;
    }

    /* TAGS */
    $tags = '';
    if ($is_singular) {
        $tag_names = wp_get_post_terms($post->ID, 'post_tag', ['fields' => 'names']);
        if (!is_wp_error($tag_names) && !empty($tag_names)) {
            $tags = implode(', ', $tag_names);
        }
    }

    /* CATEGORIES */
    $cats = '';
    if ($is_singular) {
        $cat_names = wp_get_post_terms($post->ID, 'category', ['fields' => 'names']);
        if (!is_wp_error($cat_names) && !empty($cat_names)) {
            $cats = implode(', ', $cat_names);
        }
    }

    $og_type = (is_front_page() || is_home()) ? 'website' : 'article';

    /* Dates Article */
    $published_iso = $is_singular ? get_post_time('c', true, $post) : '';
    $modified_iso  = $is_singular ? get_post_modified_time('c', true, $post) : '';

    echo "\n<!-- MTC Meta - Extention F4HXN - Start -->\n";

    /* Canonical */
    echo '<link rel="canonical" href="' . esc_url($url) . "\" />\n";

    /* Home */
    echo '<meta name="homepage" content="' . esc_url($home_url) . "\" />\n";

    /* SEO */
    echo '<meta name="description" content="' . esc_attr($description) . "\" />\n";
    echo '<meta name="keywords" content="' . esc_attr($keywords) . "\" />\n";

    /* Tags / Cats */
    if ($tags) echo '<meta name="post-tags" content="' . esc_attr($tags) . "\" />\n";
    if ($cats) echo '<meta name="post-categories" content="' . esc_attr($cats) . "\" />\n";

    /* OpenGraph */
    echo '<meta property="og:type" content="' . esc_attr($og_type) . "\" />\n";
    echo '<meta property="og:site_name" content="' . esc_attr($site_name) . "\" />\n";
    echo '<meta property="og:title" content="' . esc_attr($title) . "\" />\n";
    echo '<meta property="og:description" content="' . esc_attr($description) . "\" />\n";
    echo '<meta property="og:url" content="' . esc_url($url) . "\" />\n";
    if ($image) echo '<meta property="og:image" content="' . esc_url($image) . "\" />\n";
    echo '<meta property="og:locale" content="' . esc_attr($locale) . "\" />\n";

    /* Article times for OG */
    if ($is_singular && $published_iso) {
        echo '<meta property="article:published_time" content="' . esc_attr($published_iso) . "\" />\n";
    }
    if ($is_singular && $modified_iso) {
        echo '<meta property="article:modified_time" content="' . esc_attr($modified_iso) . "\" />\n";
    }

    /* Twitter */
    echo '<meta name="twitter:card" content="summary_large_image" />' . "\n";
    echo '<meta name="twitter:title" content="' . esc_attr($title) . "\" />\n";
    echo '<meta name="twitter:description" content="' . esc_attr($description) . "\" />\n";
    if ($image) echo '<meta name="twitter:image" content="' . esc_url($image) . "\" />\n";

    echo "<!-- MTC Meta End -->\n";

    /* JSON-LD */
    $jsonld = [];

    if (is_front_page() || is_home()) {
        $website = [
            '@context' => 'https://schema.org',
            '@type'    => 'WebSite',
            'url'      => $home_url,
            'name'     => $site_name,
            'description' => $description,
            'potentialAction' => [
                '@type' => 'SearchAction',
                'target' => $home_url . '?s={search_term_string}',
                'query-input' => 'required name=search_term_string'
            ]
        ];

        // Person (Jean-Paul) en tant que propriétaire du site (facultatif mais utile)
        $person = [
            '@context' => 'https://schema.org',
            '@type'    => 'Person',
            'name'     => 'Jean-Paul Mansouri',
            'alternateName' => 'F4HXN',
            'url'      => $home_url
        ];

        $jsonld[] = $website;
        $jsonld[] = $person;

    } elseif ($is_singular) {
        $author_name = get_the_author_meta('display_name', $post->post_author);
        if (!$author_name) $author_name = 'Jean-Paul Mansouri';

        $article = [
            '@context' => 'https://schema.org',
            '@type'    => 'Article',
            'headline' => wp_strip_all_tags($title),
            'description' => $description,
            'datePublished' => $published_iso ?: null,
            'dateModified'  => $modified_iso ?: null,
            'mainEntityOfPage' => $url,
            'author' => [
                '@type' => 'Person',
                'name'  => $author_name
            ],
            'publisher' => [
                '@type' => 'Organization',
                'name'  => $site_name
            ],
            'url' => $url
        ];
        if ($image) {
            $article['image'] = [
                '@type' => 'ImageObject',
                'url'   => $image
            ];
        }
        $jsonld[] = $article;
    }

    if (!empty($jsonld)) {
        // Encodage JSON-LD sûr
        echo "<script type=\"application/ld+json\">" . wp_json_encode($jsonld, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) . "</script>\n";
    }
}
